<?php
require_once '../includes/auth_check.php';
require_once '../includes/db_connect.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

try {
    // Get JSON data
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);

    if (!$data) {
        throw new Exception('Invalid JSON data');
    }

    // Validate required fields
    if (empty($data['customer_name']) || empty($data['payment_method']) || empty($data['items'])) {
        throw new Exception('Missing required fields');
    }

    $pdo->beginTransaction();

    // Insert sale record
    $stmt = $pdo->prepare("
        INSERT INTO sales (
            customer_name, customer_phone, payment_method, 
            total_amount, notes, created_by, created_at
        ) VALUES (?, ?, ?, ?, ?, ?, NOW())
    ");

    $stmt->execute([
        $data['customer_name'],
        $data['customer_phone'] ?? null,
        $data['payment_method'],
        $data['total_amount'],
        $data['notes'] ?? null,
        $_SESSION['user_id']
    ]);

    $sale_id = $pdo->lastInsertId();

    // Process each item
    foreach ($data['items'] as $item) {
        // Insert sale item
        $stmt = $pdo->prepare("
            INSERT INTO sale_items (
                sale_id, product_id, quantity, unit_price, total_price
            ) VALUES (?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $sale_id,
            $item['product_id'],
            $item['quantity'],
            $item['unit_price'],
            $item['total_price']
        ]);

        // Update product quantity
        $stmt = $pdo->prepare("
            UPDATE products 
            SET quantity = quantity - ? 
            WHERE id = ?
        ");
        $stmt->execute([$item['quantity'], $item['product_id']]);

        // Update serial numbers if provided
        if (!empty($item['serial_numbers'])) {
            $serial_numbers = array_map('trim', explode(',', $item['serial_numbers']));
            $serial_numbers = array_filter($serial_numbers); // Remove empty values
            
            if (!empty($serial_numbers)) {
                foreach ($serial_numbers as $serial) {
                    updateSerialNumberStatus($item['product_id'], $serial, 'sold', $sale_id);
                }
            }
        }
    }

    $pdo->commit();
    echo json_encode(['success' => true]);

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
} 